package vtk;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;

/**
 * Enum used to load native library more easily. If you don't want to set the
 * specific environment variable you can provide the path of the directory that
 * contains the VTK library through a Java Property. Like in the following
 * command line:
 *
 * > java -cp vtk.jar -Dvtk.lib.dir=.../vtk-libs vtk.sample.SimpleVTK
 *
 * The directory .../vtk-libs must contain the so/dll/dylib + the jnilib files
 *
 * @author sebastien jourdain - sebastien.jourdain@kitware.com
 */
public enum vtkNativeLibrary {

    JAWT("jawt", true),//
    VTKACCELERATORSVTKM("vtkAcceleratorsVTKmJava", false),
    VTKCHARTSCORE81("vtkChartsCoreJava", true),
    VTKRENDERINGCONTEXT2DJAVA("vtkRenderingContext2DJava", true),//
    VTKCOMMONCOREJAVA("vtkCommonCoreJava", true),//
    VTKWRAPPINGJAVA81("vtkWrappingJava-8.1", true),//
    VTKCHARTSCORE("vtkChartsCoreJava", true),
    VTKCOMMONCOLOR("vtkCommonColorJava", true),
    VTKCOMMONCOMPUTATIONALGEOMETRY("vtkCommonComputationalGeometryJava", true),
    VTKCOMMONCORE("vtkCommonCoreJava", true),
    VTKCOMMONDATAMODEL("vtkCommonDataModelJava", true),
    VTKCOMMONEXECUTIONMODEL("vtkCommonExecutionModelJava", true),
    VTKCOMMONMATH("vtkCommonMathJava", true),
    VTKCOMMONMISC("vtkCommonMiscJava", true),
    VTKCOMMONSYSTEM("vtkCommonSystemJava", true),
    VTKCOMMONTRANSFORMS("vtkCommonTransformsJava", true),
    VTKDOMAINSCHEMISTRY("vtkDomainsChemistryJava", true),
    VTKDOMAINSCHEMISTRYOPENGL2("vtkDomainsChemistryOpenGL2Java", true),
    VTKDOMAINSMICROSCOPY("vtkDomainsMicroscopyJava", false),
    VTKFILTERSAMR("vtkFiltersAMRJava", true),
    VTKFILTERSCORE("vtkFiltersCoreJava", true),
    VTKFILTERSEXTRACTION("vtkFiltersExtractionJava", true),
    VTKFILTERSFLOWPATHS("vtkFiltersFlowPathsJava", true),
    VTKFILTERSGENERAL("vtkFiltersGeneralJava", true),
    VTKFILTERSGENERIC("vtkFiltersGenericJava", true),
    VTKFILTERSGEOMETRY("vtkFiltersGeometryJava", true),
    VTKFILTERSHYBRID("vtkFiltersHybridJava", true),
    VTKFILTERSHYPERTREE("vtkFiltersHyperTreeJava", true),
    VTKFILTERSIMAGING("vtkFiltersImagingJava", true),
    VTKFILTERSMODELING("vtkFiltersModelingJava", true),
    VTKFILTERSOPENTURNS("vtkFiltersOpenTurnsJava", false),
    VTKFILTERSPARALLEL("vtkFiltersParallelJava", true),
    VTKFILTERSPARALLELFLOWPATHS("vtkFiltersParallelFlowPathsJava", false),
    VTKFILTERSPARALLELGEOMETRY("vtkFiltersParallelGeometryJava", false),
    VTKFILTERSPARALLELIMAGING("vtkFiltersParallelImagingJava", true),
    VTKFILTERSPARALLELMPI("vtkFiltersParallelMPIJava", false),
    VTKFILTERSPARALLELSTATISTICS("vtkFiltersParallelStatisticsJava", false),
    VTKFILTERSPARALLELVERDICT("vtkFiltersParallelVerdictJava", false),
    VTKFILTERSPOINTS("vtkFiltersPointsJava", true),
    VTKFILTERSPROGRAMMABLE("vtkFiltersProgrammableJava", true),
    VTKFILTERSREEBGRAPH("vtkFiltersReebGraphJava", false),
    VTKFILTERSSMP("vtkFiltersSMPJava", true),
    VTKFILTERSSELECTION("vtkFiltersSelectionJava", true),
    VTKFILTERSSOURCES("vtkFiltersSourcesJava", true),
    VTKFILTERSSTATISTICS("vtkFiltersStatisticsJava", true),
    VTKFILTERSTEXTURE("vtkFiltersTextureJava", true),
    VTKFILTERSTOPOLOGY("vtkFiltersTopologyJava", true),
    VTKFILTERSVERDICT("vtkFiltersVerdictJava", true),
    VTKGEOVISCORE("vtkGeovisCoreJava", true),
    VTKIOADIOS("vtkIOADIOSJava", false),
    VTKIOAMR("vtkIOAMRJava", true),
    VTKIOCORE("vtkIOCoreJava", true),
    VTKIOENSIGHT("vtkIOEnSightJava", true),
    VTKIOEXODUS("vtkIOExodusJava", true),
    VTKIOEXPORT("vtkIOExportJava", true),
    VTKIOEXPORTOPENGL("vtkIOExportOpenGLJava", false),
    VTKIOEXPORTOPENGL2("vtkIOExportOpenGL2Java", true),
    VTKIOFFMPEG("vtkIOFFMPEGJava", false),
    VTKIOGDAL("vtkIOGDALJava", false),
    VTKIOGEOJSON("vtkIOGeoJSONJava", false),
    VTKIOGEOMETRY("vtkIOGeometryJava", true),
    VTKIOIMAGE("vtkIOImageJava", true),
    VTKIOIMPORT("vtkIOImportJava", true),
    VTKIOINFOVIS("vtkIOInfovisJava", true),
    VTKIOLAS("vtkIOLASJava", false),
    VTKIOLSDYNA("vtkIOLSDynaJava", true),
    VTKIOLEGACY("vtkIOLegacyJava", true),
    VTKIOMINC("vtkIOMINCJava", true),
    VTKIOMPIIMAGE("vtkIOMPIImageJava", false),
    VTKIOMPIPARALLEL("vtkIOMPIParallelJava", false),
    VTKIOMOVIE("vtkIOMovieJava", true),
    VTKIOMYSQL("vtkIOMySQLJava", false),
    VTKIONETCDF("vtkIONetCDFJava", true),
    VTKIOODBC("vtkIOODBCJava", false),
    VTKIOPDAL("vtkIOPDALJava", false),
    VTKIOPLY("vtkIOPLYJava", true),
    VTKIOPARALLEL("vtkIOParallelJava", true),
    VTKIOPARALLELEXODUS("vtkIOParallelExodusJava", false),
    VTKIOPARALLELLSDYNA("vtkIOParallelLSDynaJava", false),
    VTKIOPARALLELNETCDF("vtkIOParallelNetCDFJava", false),
    VTKIOPARALLELXML("vtkIOParallelXMLJava", true),
    VTKIOPARALLELXDMF3("vtkIOParallelXdmf3Java", false),
    VTKIOPOSTGRESQL("vtkIOPostgreSQLJava", false),
    VTKIOSQL("vtkIOSQLJava", true),
    VTKIOSEGY("vtkIOSegYJava", false),
    VTKIOTRUCHAS("vtkIOTRUCHASJava", false),
    VTKIOTECPLOTTABLE("vtkIOTecplotTableJava", true),
    VTKIOVPIC("vtkIOVPICJava", false),
    VTKIOVIDEO("vtkIOVideoJava", true),
    VTKIOXML("vtkIOXMLJava", true),
    VTKIOXMLPARSER("vtkIOXMLParserJava", true),
    VTKIOXDMF2("vtkIOXdmf2Java", false),
    VTKIOXDMF3("vtkIOXdmf3Java", false),
    VTKIMAGINGCOLOR("vtkImagingColorJava", true),
    VTKIMAGINGCORE("vtkImagingCoreJava", true),
    VTKIMAGINGFOURIER("vtkImagingFourierJava", true),
    VTKIMAGINGGENERAL("vtkImagingGeneralJava", true),
    VTKIMAGINGHYBRID("vtkImagingHybridJava", true),
    VTKIMAGINGMATH("vtkImagingMathJava", true),
    VTKIMAGINGMORPHOLOGICAL("vtkImagingMorphologicalJava", true),
    VTKIMAGINGOPENGL2("vtkImagingOpenGL2Java", false),
    VTKIMAGINGSOURCES("vtkImagingSourcesJava", true),
    VTKIMAGINGSTATISTICS("vtkImagingStatisticsJava", true),
    VTKIMAGINGSTENCIL("vtkImagingStencilJava", true),
    VTKINFOVISBOOSTGRAPHALGORITHMS("vtkInfovisBoostGraphAlgorithmsJava", false),
    VTKINFOVISCORE("vtkInfovisCoreJava", true),
    VTKINFOVISLAYOUT("vtkInfovisLayoutJava", true),
    VTKINTERACTIONIMAGE("vtkInteractionImageJava", true),
    VTKINTERACTIONSTYLE("vtkInteractionStyleJava", true),
    VTKINTERACTIONWIDGETS("vtkInteractionWidgetsJava", true),
    VTKPARALLELCORE("vtkParallelCoreJava", true),
    VTKPARALLELMPI("vtkParallelMPIJava", false),
    VTKRENDERINGANNOTATION("vtkRenderingAnnotationJava", true),
    VTKRENDERINGCONTEXT2D("vtkRenderingContext2DJava", true),
    VTKRENDERINGCONTEXTOPENGL("vtkRenderingContextOpenGLJava", false),
    VTKRENDERINGCONTEXTOPENGL2("vtkRenderingContextOpenGL2Java", true),
    VTKRENDERINGCORE("vtkRenderingCoreJava", true),
    VTKRENDERINGEXTERNAL("vtkRenderingExternalJava", false),
    VTKRENDERINGFREETYPE("vtkRenderingFreeTypeJava", true),
    VTKRENDERINGGL2PS("vtkRenderingGL2PSJava", false),
    VTKRENDERINGGL2PSOPENGL2("vtkRenderingGL2PSOpenGL2Java", true),
    VTKRENDERINGIMAGE("vtkRenderingImageJava", true),
    VTKRENDERINGLIC("vtkRenderingLICJava", false),
    VTKRENDERINGLICOPENGL2("vtkRenderingLICOpenGL2Java", false),
    VTKRENDERINGLOD("vtkRenderingLODJava", true),
    VTKRENDERINGLABEL("vtkRenderingLabelJava", true),
    VTKRENDERINGMATPLOTLIB("vtkRenderingMatplotlibJava", false),
    VTKRENDERINGOSPRAY("vtkRenderingOSPRayJava", false),
    VTKRENDERINGOCULUS("vtkRenderingOculusJava", false),
    VTKRENDERINGOPENGL("vtkRenderingOpenGLJava", false),
    VTKRENDERINGOPENGL2("vtkRenderingOpenGL2Java", true),
    VTKRENDERINGOPENVR("vtkRenderingOpenVRJava", false),
    VTKRENDERINGOPTIX("vtkRenderingOptiXJava", false),
    VTKRENDERINGPARALLEL("vtkRenderingParallelJava", false),
    VTKRENDERINGPARALLELLIC("vtkRenderingParallelLICJava", false),
    VTKRENDERINGQT("vtkRenderingQtJava", false),
    VTKRENDERINGSCENEGRAPH("vtkRenderingSceneGraphJava", false),
    VTKRENDERINGVOLUME("vtkRenderingVolumeJava", true),
    VTKRENDERINGVOLUMEAMR("vtkRenderingVolumeAMRJava", false),
    VTKRENDERINGVOLUMEOPENGL("vtkRenderingVolumeOpenGLJava", false),
    VTKRENDERINGVOLUMEOPENGL2("vtkRenderingVolumeOpenGL2Java", true),
    VTKTESTINGRENDERING("vtkTestingRenderingJava", false),
    VTKPYTHONINTERPRETER("vtkPythonInterpreterJava", false),
    VTKVIEWSCONTEXT2D("vtkViewsContext2DJava", true),
    VTKVIEWSCORE("vtkViewsCoreJava", true),
    VTKVIEWSGEOVIS("vtkViewsGeovisJava", false),
    VTKVIEWSINFOVIS("vtkViewsInfovisJava", true),
    VTKWEBCORE("vtkWebCoreJava", false),
    VTKWEBGLEXPORTER("vtkWebGLExporterJava", false);


  /**
   * Try to load all library
   *
   * @return true if all library have been successfully loaded
   */
  public static boolean LoadAllNativeLibraries() {
    extractAllNativeLibraries();
    boolean isEveryThingLoaded = true;
    for (vtkNativeLibrary lib : values()) {
      try {
        if(lib.IsBuilt()) {
          lib.LoadLibrary();
        }
      } catch (UnsatisfiedLinkError e) {
        isEveryThingLoaded = false;
        e.printStackTrace();
      }
    }

    return isEveryThingLoaded;
  }

  /**
   * Load the set of given library and trows runtime exception if any given
   * library failed in the loading process
   *
   * @param nativeLibraries
   */
  public static void LoadNativeLibraries(vtkNativeLibrary... nativeLibraries) {
    for (vtkNativeLibrary lib : nativeLibraries) {
      lib.LoadLibrary();
    }
  }

  /**
   * Disable the pop-in vtkErrorWindow by writing the error to a log file.
   * If the provided logFile is null the default "vtkError.txt" file will be
   * used.
   *
   * @param logFile
   */
  public static void DisableOutputWindow(File logFile) {
    if(logFile == null) {
      logFile = new File("vtkError.txt");
    }
    vtkFileOutputWindow outputError = new vtkFileOutputWindow();
    outputError.SetFileName(logFile.getAbsolutePath());
    outputError.SetInstance(outputError);
  }

  private vtkNativeLibrary(String nativeLibraryName, boolean built) {
    this.nativeLibraryName = nativeLibraryName;
    this.loaded = false;
    this.built = built;
  }

  /**
   * Load the library and throws runtime exception if the library failed in
   * the loading process
   */
  public void LoadLibrary() throws UnsatisfiedLinkError {
    if (!loaded) {
      if (System.getProperty("vtk.lib.dir") != null) {
        File dir = new File(System.getProperty("vtk.lib.dir"));
        patchJavaLibraryPath(dir.getAbsolutePath());
        File libPath = new File(dir, System.mapLibraryName(nativeLibraryName));
        if (libPath.exists()) {
          try {
            Runtime.getRuntime().load(libPath.getAbsolutePath());
            loaded = true;
            return;
          } catch (UnsatisfiedLinkError e) {
            e.printStackTrace();
          }
        }
      }
      System.loadLibrary(nativeLibraryName);
    }
    loaded = true;
  }

  /**
   * @return true if the library has already been successfully loaded
   */
  public boolean IsLoaded() {
    return loaded;
  }

  /**
   * @return true if the module was enabled and therefore build
   */
  public boolean IsBuilt() {
    return built;
  }

  /**
   * @return the library name
   */
  public String GetLibraryName() {
    return nativeLibraryName;
  }

  private static void patchJavaLibraryPath(String vtkLibDir) {
    if (vtkLibDir != null) {
      String path_separator = System.getProperty("path.separator");
      String s = System.getProperty("java.library.path");
      if (!s.contains(vtkLibDir)) {
        s = s + path_separator + vtkLibDir;
        System.setProperty("java.library.path", s);
      }
    }
  }

  private String nativeLibraryName;
  private boolean loaded;
  private boolean built;
  
  /**
   * Try to extract all library native files to [user]/.insane/lib dir
   *
   * @return true if all library have been successfully extracted
   */
  private static void extractAllNativeLibraries() {
    
    try {
      String tempDirectory = System.getProperty("user.home", "./" + "tmplib");
      File tmpDir = new File(tempDirectory);
      if (!tmpDir.isDirectory()) {
        tmpDir.mkdirs();
        if (!tmpDir.isDirectory())
          throw new IOException("Unable to create temporary directory " + tmpDir);
      }
      
      String nativeDestDirectory = tempDirectory + File.separator + ".vtk";
      File vtkNativeDir = new File(nativeDestDirectory);
      if (!vtkNativeDir.isDirectory()) {
        vtkNativeDir.mkdirs();
        if (!vtkNativeDir.isDirectory())
          throw new IOException("Unable to create temporary directory " + vtkNativeDir);
      }
      
      //Sets nativeSourceDirectory
      String nativeSourceDirectory = "natives/linux_64/";
      if (OsHelper.isWindows()) {
        nativeSourceDirectory = "natives/windows_64/";
      } else if (OsHelper.isMac()) {
        nativeSourceDirectory = "natives/mac_64/";
      }
      
      //Gets ClassLoader
      ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
      if (classLoader == null) {
        classLoader = Class.class.getClassLoader();
      }

      //Reads list of files to be extracted from FilesList.txt
      BufferedReader listFile = new BufferedReader(
          new InputStreamReader(classLoader.getResourceAsStream(nativeSourceDirectory + "FilesList.txt"),StandardCharsets.UTF_8)
          );
      String nativefilename;
      while ((nativefilename = listFile.readLine()) != null) {
        extractFile(nativeSourceDirectory + nativefilename, nativeDestDirectory + File.separator + nativefilename);
      }
      
      // corrects Java Library Path 
      addToJavaLibraryPath(nativeDestDirectory);
      
    } catch (Exception e) {
      e.printStackTrace();
      System.exit(1);
    }
  }
  
  private static void extractFile(String sourcePath, String destPath) throws IOException, FileNotFoundException, Exception {
    
    //Getting ClassLoader
    ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
    if (classLoader == null) {
      classLoader = Class.class.getClassLoader();
    }
    
    if (!new File(destPath).exists()) {
      InputStream inputStream = classLoader.getResourceAsStream(sourcePath);
      
      // copy resource stream to temporary file
      OutputStream outputStream = new FileOutputStream(destPath);
      copyBetweenStreams(inputStream, outputStream);
      outputStream.close();
      inputStream.close();
      
      // note that this doesn't always work:/
      File outfile = new File(sourcePath);
      outfile.deleteOnExit();
    }
  }
  
  private static void copyBetweenStreams( InputStream inputStream, OutputStream outputStream ) throws Exception {
    byte[] buffer = new byte[1000000];
    int len;
    while ((len = inputStream.read(buffer)) != -1) {
        outputStream.write(buffer, 0, len);
    }
  }
  
  public static final void addToJavaLibraryPath(String newpath) throws Exception {
    String oldLibraryPath = System.getProperty("java.library.path");
    if (oldLibraryPath != null) {
      System.setProperty("java.library.path", oldLibraryPath + File.pathSeparator + newpath);
    } else {
      System.setProperty("java.library.path", newpath);
    }
    //java.library.path is a ready-only system variable in what concerns native library loading
    //the code below is a hack which forces this variable to be reevaluated
    //Field fieldSysPath = ClassLoader.class.getDeclaredField( "sys_paths" );
    //fieldSysPath.setAccessible( true );
    //fieldSysPath.set( null, null );
  }
  
  private static class OsHelper {
    public static final int jvmBits() {
        if( System.getProperty("os.arch").toLowerCase().equals("x86") ) {
             return 32;
        }
        if( System.getProperty("os.arch").toLowerCase().equals("i386") ) {
             return 32;
        }
        return 64;
    }
    public static final boolean isLinux() {
      return (System.getProperty("os.name").toLowerCase().indexOf("nix") >= 0 ||
          System.getProperty("os.name").toLowerCase().indexOf("nux") >= 0 ||
          System.getProperty("os.name").toLowerCase().indexOf("aix") >= 0);
    }
    public static final boolean isWindows() {
        return System.getProperty("os.name").toLowerCase().indexOf("win") >= 0;
    }
    public static final boolean isMac() {
        return System.getProperty("os.name").toLowerCase().indexOf("mac") >= 0;
    }
}
  
}
