package CGAL;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

public class NativeHelper {

  private static boolean osNativeLoaded = false;

  /**
   * Extracts the native library to [user]/.cgal dir and loads it
   * @param libraryName a String representing the library name.
   */
  public static void loadNativeLibrary(String libraryName) {

    try {
      String tempDirectory = System.getProperty("user.home", "./" + "tmplib");
      File tmpDir = new File(tempDirectory);
      if (!tmpDir.isDirectory()) {
        tmpDir.mkdirs();
        if (!tmpDir.isDirectory())
          throw new IOException("Unable to create temporary directory " + tmpDir);
      }

      String nativeDestDirectory = tempDirectory + File.separator + ".cgal";
      File cgalNativeDir = new File(nativeDestDirectory);
      if (!cgalNativeDir.isDirectory()) {
        cgalNativeDir.mkdirs();
        if (!cgalNativeDir.isDirectory())
          throw new IOException("Unable to create temporary directory " + cgalNativeDir);
      }

      // Sets nativeSourceDirectory
      String nativeSourceDirectory = "natives/linux_64/";
      if (OsHelper.isWindows()) {
        nativeSourceDirectory = "natives/windows_64/";
      } else if (OsHelper.isMac()) {
        nativeSourceDirectory = "natives/mac_64/";
      }

      // Extract and load OS specific dependencies only first time this method is executed
      if (!osNativeLoaded) {
        if (OsHelper.isLinux()) {
          extractAndLoadFile("libboost_system.so.1.58.0", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libboost_thread.so.1.58.0", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libgmp.so", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libmpfr.so.4", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libmpfr.so", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libCGAL.so.13", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libCGAL.so", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libCGAL_Core.so", nativeSourceDirectory, nativeDestDirectory);
          // extractAndLoadFile("libCGAL_ImageIO.so", nativeSourceDirectory, nativeDestDirectory);
          // missing libzlib.so -> dependency for CGAL_ImageIO.so
        } else if (OsHelper.isWindows()) {
          extractAndLoadFile("msvcp140.dll", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("vcruntime140.dll", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("vcruntime140_1.dll", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libgmp-10.dll", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("libmpfr-4.dll", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("CGAL-vc142-mt-5.0.dll", nativeSourceDirectory, nativeDestDirectory);
          extractAndLoadFile("CGAL_Core-vc142-mt-5.0.dll", nativeSourceDirectory, nativeDestDirectory);
          // extractAndLoadFile("CGAL_ImageIO-vc142-mt-5.0.dll", nativeSourceDirectory, nativeDestDirectory);
          // missing zlib.dll -> dependency for CGAL_ImageIO-vc142-mt-5.0.dll
        } else if (OsHelper.isMac()) {
          // to complete if native libraries are compiled for mac
        }
        osNativeLoaded = true;
      }

      // Extract and load library
      extractAndLoadLibrary(libraryName, nativeSourceDirectory, nativeDestDirectory);

    } catch (Exception e) {
      e.printStackTrace();
      System.exit(1);
    }
  }

  private static void extractAndLoadFile(String fileName, String sourceDirectory, String destDirectory) throws IOException, FileNotFoundException, Exception {
    extractFile(sourceDirectory + fileName, destDirectory + File.separator + fileName);
    File libPath = new File(destDirectory, fileName);
    Runtime.getRuntime().load(libPath.getAbsolutePath());
  }

  private static void extractAndLoadLibrary(String fileName, String sourceDirectory, String destDirectory) throws IOException, FileNotFoundException, Exception {
    extractAndLoadFile(System.mapLibraryName(fileName), sourceDirectory, destDirectory);
  }

  private static void extractFile(String sourcePath, String destPath) throws IOException, FileNotFoundException, Exception {
    // Getting ClassLoader
    ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
    if (classLoader == null) {
      classLoader = Class.class.getClassLoader();
    }

    if (!new File(destPath).exists()) {
      InputStream inputStream = classLoader.getResourceAsStream(sourcePath);

      // copy resource stream to temporary file
      OutputStream outputStream = new FileOutputStream(destPath);
      copyBetweenStreams(inputStream, outputStream);
      outputStream.close();
      inputStream.close();

      // note that this doesn't always work:/
      File outfile = new File(destPath);
      outfile.deleteOnExit();
    }
  }

  private static void copyBetweenStreams(InputStream inputStream, OutputStream outputStream) throws Exception {
    byte[] buffer = new byte[1000000];
    int len;
    while ((len = inputStream.read(buffer)) != -1) {
      outputStream.write(buffer, 0, len);
    }
  }

  /**
   * Inner class to help discover current OS.
   */
  private static class OsHelper {
    public static final int jvmBits() {
      if (System.getProperty("os.arch").toLowerCase().equals("x86")) {
        return 32;
      }
      if (System.getProperty("os.arch").toLowerCase().equals("i386")) {
        return 32;
      }
      return 64;
    }

    public static final boolean isLinux() {
      return (System.getProperty("os.name").toLowerCase().indexOf("nix") >= 0 || System.getProperty("os.name").toLowerCase().indexOf("nux") >= 0
          || System.getProperty("os.name").toLowerCase().indexOf("aix") >= 0);
    }

    public static final boolean isWindows() {
      return System.getProperty("os.name").toLowerCase().indexOf("win") >= 0;
    }

    public static final boolean isMac() {
      return System.getProperty("os.name").toLowerCase().indexOf("mac") >= 0;
    }
  }

}
